<?php
require_once 'config/config.php';

// Redirect if already logged in
if (isLoggedIn()) {
    $role = getCurrentUserRole();
    redirect($role . '/dashboard.php');
}

$pageTitle = 'Register - PSAU GAD';
$error = '';
$success = '';

if (isPost()) {
    $email = sanitize(post('email'));
    $password = post('password');
    $confirmPassword = post('confirm_password');
    $firstName = sanitize(post('first_name'));
    $lastName = sanitize(post('last_name'));
    $middleName = sanitize(post('middle_name'));
    $gender = sanitize(post('gender'));
    $category = sanitize(post('category'));
    
    // Validation
    if (empty($email) || empty($password) || empty($firstName) || empty($lastName) || empty($gender) || empty($category)) {
        $error = 'Please fill in all required fields';
    } elseif ($password !== $confirmPassword) {
        $error = 'Passwords do not match';
    } elseif (strlen($password) < PASSWORD_MIN_LENGTH) {
        $error = 'Password must be at least ' . PASSWORD_MIN_LENGTH . ' characters';
    } else {
        $auth = new Auth();
        
        // Check if PSAU email
        $isPsauAccount = $auth->isPsauEmail($email);
        
        // Register user
        $result = $auth->register($email, $password, 'employee', $isPsauAccount);
        
        if ($result['success']) {
            $userId = $result['user_id'];
            
            // Create profile
            $profileData = [
                'first_name' => $firstName,
                'middle_name' => $middleName,
                'last_name' => $lastName,
                'gender' => $gender,
                'category' => $category
            ];
            
            $auth->updateProfile($userId, $profileData);
            
            if ($isPsauAccount) {
                $success = 'Registration successful! You can now login.';
            } else {
                $success = 'Registration successful! Please wait for admin approval before logging in.';
            }
            
            // Clear form
            $email = $firstName = $lastName = $middleName = $gender = $category = '';
        } else {
            $error = $result['message'];
        }
    }
}

include 'includes/header.php';
?>

<div class="min-h-screen bg-gradient-to-br from-purple-50 to-purple-100 py-12 px-4 sm:px-6 lg:px-8">
    <div class="max-w-2xl mx-auto">
        <!-- Logo and Title -->
        <div class="text-center mb-8">
            <div class="mx-auto w-20 h-20 bg-primary rounded-full flex items-center justify-center mb-4">
                <i class="fas fa-user-plus text-white text-3xl"></i>
            </div>
            <h2 class="text-3xl font-bold text-gray-900">Create Account</h2>
            <p class="mt-2 text-gray-600">Join PSAU GAD Information System</p>
        </div>
        
        <!-- Registration Form -->
        <div class="bg-white rounded-2xl shadow-xl p-8">
            <?php if ($error): ?>
                <div class="mb-4 bg-red-50 border-l-4 border-red-500 p-4 rounded">
                    <p class="text-red-700"><i class="fas fa-exclamation-circle mr-2"></i><?php echo $error; ?></p>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="mb-4 bg-green-50 border-l-4 border-green-500 p-4 rounded">
                    <p class="text-green-700"><i class="fas fa-check-circle mr-2"></i><?php echo $success; ?></p>
                    <a href="login.php" class="text-green-800 font-semibold underline mt-2 inline-block">
                        Go to Login <i class="fas fa-arrow-right ml-1"></i>
                    </a>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="" class="space-y-6">
                <!-- Personal Information -->
                <div>
                    <h3 class="text-lg font-semibold text-gray-900 mb-4 border-b pb-2">
                        <i class="fas fa-user mr-2 text-primary"></i> Personal Information
                    </h3>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <!-- First Name -->
                        <div>
                            <label for="first_name" class="block text-sm font-medium text-gray-700 mb-2">
                                First Name <span class="text-red-500">*</span>
                            </label>
                            <input type="text" 
                                   id="first_name" 
                                   name="first_name" 
                                   required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                                   value="<?php echo htmlspecialchars($firstName ?? ''); ?>">
                        </div>
                        
                        <!-- Last Name -->
                        <div>
                            <label for="last_name" class="block text-sm font-medium text-gray-700 mb-2">
                                Last Name <span class="text-red-500">*</span>
                            </label>
                            <input type="text" 
                                   id="last_name" 
                                   name="last_name" 
                                   required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                                   value="<?php echo htmlspecialchars($lastName ?? ''); ?>">
                        </div>
                        
                        <!-- Middle Name -->
                        <div>
                            <label for="middle_name" class="block text-sm font-medium text-gray-700 mb-2">
                                Middle Name
                            </label>
                            <input type="text" 
                                   id="middle_name" 
                                   name="middle_name"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                                   value="<?php echo htmlspecialchars($middleName ?? ''); ?>">
                        </div>
                        
                        <!-- Gender -->
                        <div>
                            <label for="gender" class="block text-sm font-medium text-gray-700 mb-2">
                                Gender <span class="text-red-500">*</span>
                            </label>
                            <select id="gender" 
                                    name="gender" 
                                    required
                                    class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                                <option value="">Select Gender</option>
                                <option value="Male" <?php echo (isset($gender) && $gender === 'Male') ? 'selected' : ''; ?>>Male</option>
                                <option value="Female" <?php echo (isset($gender) && $gender === 'Female') ? 'selected' : ''; ?>>Female</option>
                                <option value="Other" <?php echo (isset($gender) && $gender === 'Other') ? 'selected' : ''; ?>>Other</option>
                            </select>
                        </div>
                        
                        <!-- Category -->
                        <div class="md:col-span-2">
                            <label for="category" class="block text-sm font-medium text-gray-700 mb-2">
                                Category <span class="text-red-500">*</span>
                            </label>
                            <select id="category" 
                                    name="category" 
                                    required
                                    class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                                <option value="">Select Category</option>
                                <option value="Faculty" <?php echo (isset($category) && $category === 'Faculty') ? 'selected' : ''; ?>>Faculty</option>
                                <option value="Non-Teaching" <?php echo (isset($category) && $category === 'Non-Teaching') ? 'selected' : ''; ?>>Non-Teaching</option>
                                <option value="Service Personnel" <?php echo (isset($category) && $category === 'Service Personnel') ? 'selected' : ''; ?>>Service Personnel</option>
                                <option value="Other" <?php echo (isset($category) && $category === 'Other') ? 'selected' : ''; ?>>Other</option>
                            </select>
                        </div>
                    </div>
                </div>
                
                <!-- Account Information -->
                <div>
                    <h3 class="text-lg font-semibold text-gray-900 mb-4 border-b pb-2">
                        <i class="fas fa-lock mr-2 text-primary"></i> Account Information
                    </h3>
                    
                    <div class="space-y-4">
                        <!-- Email -->
                        <div>
                            <label for="email" class="block text-sm font-medium text-gray-700 mb-2">
                                Email Address <span class="text-red-500">*</span>
                            </label>
                            <input type="email" 
                                   id="email" 
                                   name="email" 
                                   required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                                   placeholder="your.email@psau.edu.ph"
                                   value="<?php echo htmlspecialchars($email ?? ''); ?>">
                            <p class="mt-1 text-sm text-gray-500">
                                <i class="fas fa-info-circle mr-1"></i> Use your PSAU email for automatic approval
                            </p>
                        </div>
                        
                        <!-- Password -->
                        <div>
                            <label for="password" class="block text-sm font-medium text-gray-700 mb-2">
                                Password <span class="text-red-500">*</span>
                            </label>
                            <input type="password" 
                                   id="password" 
                                   name="password" 
                                   required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                                   placeholder="Minimum 8 characters">
                        </div>
                        
                        <!-- Confirm Password -->
                        <div>
                            <label for="confirm_password" class="block text-sm font-medium text-gray-700 mb-2">
                                Confirm Password <span class="text-red-500">*</span>
                            </label>
                            <input type="password" 
                                   id="confirm_password" 
                                   name="confirm_password" 
                                   required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                                   placeholder="Re-enter password">
                        </div>
                    </div>
                </div>
                
                <!-- Submit Button -->
                <button type="submit" 
                        class="w-full bg-primary text-white py-3 rounded-lg font-semibold hover:bg-primary-dark transition transform hover:scale-105 btn-animate">
                    <i class="fas fa-user-plus mr-2"></i> Create Account
                </button>
            </form>
            
            <!-- Login Link -->
            <div class="mt-6 text-center">
                <p class="text-gray-600">
                    Already have an account? 
                    <a href="login.php" class="text-primary font-semibold hover:text-primary-dark">
                        Sign in here
                    </a>
                </p>
            </div>
            
            <!-- Back to Home -->
            <div class="mt-4 text-center">
                <a href="index.php" class="text-sm text-gray-600 hover:text-primary">
                    <i class="fas fa-arrow-left mr-1"></i> Back to Home
                </a>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>

