<?php
require_once '../config/config.php';
requireRole('employee');

$pageTitle = 'My Certificates - PSAU GAD';
$db = getDB();
$userId = getCurrentUserId();

// Get all attended activities with evaluation status
$stmt = $db->prepare("
    SELECT
        ar.registration_id,
        a.activity_id,
        a.title,
        a.activity_date,
        a.category,
        a.venue,
        aa.status as attendance_status,
        aa.check_in_time,
        ef.form_id as evaluation_form_id,
        er.response_id as evaluation_submitted
    FROM activity_registrations ar
    JOIN activities a ON ar.activity_id = a.activity_id
    JOIN activity_attendance aa ON ar.registration_id = aa.registration_id
    LEFT JOIN evaluation_forms ef ON a.activity_id = ef.activity_id AND ef.is_active = 1
    LEFT JOIN evaluation_responses er ON ef.form_id = er.form_id AND er.user_id = ar.user_id
    WHERE ar.user_id = ?
    AND ar.status = 'approved'
    AND aa.status IN ('present', 'late')
    ORDER BY a.activity_date DESC
");
$stmt->execute([$userId]);
$allActivities = $stmt->fetchAll();

// Separate into available certificates and pending
$certificates = [];
$pendingCertificates = [];

foreach ($allActivities as $activity) {
    // Check if certificate is available
    if ($activity['evaluation_form_id']) {
        // Evaluation required - check if submitted
        if ($activity['evaluation_submitted']) {
            $certificates[] = $activity;
        } else {
            $pendingCertificates[] = $activity;
        }
    } else {
        // No evaluation required - certificate available
        $certificates[] = $activity;
    }
}

include '../includes/header.php';
include '../includes/navbar.php';
?>

<div class="min-h-screen bg-gray-50 py-8">
    <div class="container mx-auto px-4">
        <div class="mb-8">
            <h1 class="text-3xl font-bold text-gray-900 mb-2"><i class="fas fa-certificate text-yellow-500 mr-2"></i>My Certificates</h1>
            <p class="text-gray-600">View and download your GAD activity certificates</p>
        </div>

        <!-- Statistics -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-yellow-500">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-500 text-sm font-semibold">Total Certificates</p>
                        <h3 class="text-3xl font-bold text-gray-900 mt-2"><?php echo count($certificates); ?></h3>
                    </div>
                    <div class="bg-yellow-100 p-4 rounded-full"><i class="fas fa-certificate text-yellow-500 text-2xl"></i></div>
                </div>
            </div>
            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-blue-500">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-500 text-sm font-semibold">Pending</p>
                        <h3 class="text-3xl font-bold text-gray-900 mt-2"><?php echo count($pendingCertificates); ?></h3>
                    </div>
                    <div class="bg-blue-100 p-4 rounded-full"><i class="fas fa-clock text-blue-500 text-2xl"></i></div>
                </div>
            </div>
            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-employee">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-500 text-sm font-semibold">This Year</p>
                        <h3 class="text-3xl font-bold text-gray-900 mt-2"><?php echo count(array_filter($certificates, fn($c) => date('Y', strtotime($c['activity_date'])) == date('Y'))); ?></h3>
                    </div>
                    <div class="bg-employee bg-opacity-10 p-4 rounded-full"><i class="fas fa-calendar text-employee text-2xl"></i></div>
                </div>
            </div>
        </div>

        <!-- Pending Certificates -->
        <?php if (!empty($pendingCertificates)): ?>
            <div class="bg-yellow-50 border border-yellow-200 rounded-xl p-6 mb-8">
                <h2 class="text-xl font-bold text-yellow-900 mb-4"><i class="fas fa-exclamation-circle mr-2"></i>Complete Evaluation to Get Certificate</h2>
                <p class="text-yellow-800 mb-4">You have attended the following activities. Please submit the evaluation to receive your certificate:</p>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <?php foreach ($pendingCertificates as $activity): ?>
                        <div class="bg-white rounded-lg p-4 border border-yellow-200">
                            <h4 class="font-semibold text-gray-900 mb-2"><?php echo htmlspecialchars($activity['title']); ?></h4>
                            <div class="flex items-center text-sm text-gray-600 mb-3">
                                <i class="fas fa-calendar w-5 text-employee"></i>
                                <span><?php echo formatDate($activity['activity_date']); ?></span>
                            </div>
                            <a href="evaluation.php?activity_id=<?php echo $activity['activity_id']; ?>"
                               class="block w-full bg-pink-500 text-white text-center px-3 py-2 rounded-lg hover:bg-pink-600 transition font-semibold text-sm">
                                <i class="fas fa-star mr-2"></i>Submit Evaluation
                            </a>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Certificates Grid -->
        <?php if (empty($certificates)): ?>
            <div class="bg-white rounded-xl shadow-lg p-12 text-center">
                <i class="fas fa-certificate text-gray-300 text-6xl mb-4"></i>
                <h3 class="text-xl font-bold text-gray-900 mb-2">No Certificates Yet</h3>
                <p class="text-gray-600 mb-6">Attend GAD activities to earn certificates.</p>
                <a href="activities.php" class="inline-block bg-employee text-white px-6 py-3 rounded-lg hover:bg-green-700 transition font-semibold">
                    <i class="fas fa-calendar-plus mr-2"></i>Browse Activities
                </a>
            </div>
        <?php else: ?>
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                <?php foreach ($certificates as $cert): ?>
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden hover:shadow-xl transition">
                        <div class="bg-gradient-to-r from-yellow-400 to-yellow-600 p-6 text-center">
                            <i class="fas fa-certificate text-white text-5xl mb-3"></i>
                            <h3 class="text-white font-bold text-lg">Certificate of Participation</h3>
                        </div>
                        <div class="p-6">
                            <h4 class="font-bold text-gray-900 mb-3"><?php echo htmlspecialchars($cert['title']); ?></h4>
                            <div class="space-y-2 mb-4">
                                <div class="flex items-center text-sm text-gray-600">
                                    <i class="fas fa-calendar w-5 text-employee"></i>
                                    <span><?php echo formatDate($cert['activity_date']); ?></span>
                                </div>
                                <div class="flex items-center text-sm text-gray-600">
                                    <i class="fas fa-tag w-5 text-employee"></i>
                                    <span><?php echo htmlspecialchars($cert['category']); ?></span>
                                </div>
                                <?php if ($cert['venue']): ?>
                                    <div class="flex items-center text-sm text-gray-600">
                                        <i class="fas fa-map-marker-alt w-5 text-employee"></i>
                                        <span><?php echo htmlspecialchars($cert['venue']); ?></span>
                                    </div>
                                <?php endif; ?>
                                <div class="flex items-center text-sm text-gray-600">
                                    <i class="fas fa-fingerprint w-5 text-employee"></i>
                                    <span class="font-mono text-xs">PSAU-GAD-<?php echo str_pad($cert['registration_id'], 6, '0', STR_PAD_LEFT); ?></span>
                                </div>
                            </div>
                            <a href="certificate.php?id=<?php echo $cert['registration_id']; ?>"
                               class="block w-full bg-employee text-white text-center px-4 py-2 rounded-lg hover:bg-green-700 transition font-semibold">
                                <i class="fas fa-certificate mr-2"></i>View Certificate
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php include '../includes/footer.php'; ?>

