<?php
require_once '../config/config.php';
requireRole('admin');

$pageTitle = 'GPB Form - Admin - PSAU GAD';
$db = getDB();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_gpb'])) {
    $year = $_POST['year'];
    $activity_title = $_POST['activity_title'];
    $description = $_POST['description'];
    $target_date = $_POST['target_date'];
    $estimated_budget = $_POST['estimated_budget'];
    $target_participants = $_POST['target_participants'];
    $category = $_POST['category'];
    $created_by = getCurrentUserId();
    
    $stmt = $db->prepare("
        INSERT INTO gpb_forms (year, activity_title, description, target_date, estimated_budget, target_participants, category, created_by)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    if ($stmt->execute([$year, $activity_title, $description, $target_date, $estimated_budget, $target_participants, $category, $created_by])) {
        $success = "GPB entry created successfully!";
    } else {
        $error = "Failed to create GPB entry.";
    }
}

// Get existing GPB entries
$year_filter = $_GET['year'] ?? date('Y');
$stmt = $db->prepare("
    SELECT g.*, u.email as created_by_email
    FROM gpb_forms g
    LEFT JOIN users u ON g.created_by = u.user_id
    WHERE g.year = ?
    ORDER BY g.created_at DESC
");
$stmt->execute([$year_filter]);
$gpb_entries = $stmt->fetchAll();

// Calculate totals
$total_budget = array_sum(array_column($gpb_entries, 'estimated_budget'));
$total_participants = array_sum(array_column($gpb_entries, 'target_participants'));

include '../includes/header.php';
include '../includes/navbar.php';
?>

<div class="min-h-screen bg-gradient-to-br from-blue-50 via-white to-indigo-50 py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">
                        <i class="fas fa-file-invoice-dollar text-admin mr-3"></i>
                        GPB Form
                    </h1>
                    <p class="text-gray-600 mt-2">Gender and Development Plan and Budget</p>
                </div>
                <a href="reports.php" class="bg-gray-600 hover:bg-gray-700 text-white px-6 py-3 rounded-lg font-semibold transition btn-animate">
                    <i class="fas fa-arrow-left mr-2"></i>
                    Back to Reports
                </a>
            </div>
        </div>

        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg mb-6">
                <i class="fas fa-check-circle mr-2"></i>
                <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-6">
                <i class="fas fa-exclamation-circle mr-2"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>

        <!-- Create GPB Entry Form -->
        <div class="bg-white rounded-xl shadow-lg p-6 mb-8">
            <h2 class="text-xl font-bold text-gray-900 mb-6">
                <i class="fas fa-plus-circle text-admin mr-2"></i>
                Create New GPB Entry
            </h2>
            <form method="POST" class="space-y-6">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <!-- Year -->
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Year *</label>
                        <select name="year" required class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-admin focus:border-transparent">
                            <?php for ($y = date('Y'); $y <= date('Y') + 5; $y++): ?>
                                <option value="<?php echo $y; ?>"><?php echo $y; ?></option>
                            <?php endfor; ?>
                        </select>
                    </div>

                    <!-- Category -->
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Category *</label>
                        <select name="category" required class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-admin focus:border-transparent">
                            <option value="">Select Category</option>
                            <option value="Gender Sensitivity">Gender Sensitivity</option>
                            <option value="Women Empowerment">Women Empowerment</option>
                            <option value="LGBTQ+ Rights">LGBTQ+ Rights</option>
                            <option value="Anti-VAWC">Anti-VAWC</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>

                    <!-- Activity Title -->
                    <div class="md:col-span-2">
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Activity Title *</label>
                        <input type="text" name="activity_title" required 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-admin focus:border-transparent"
                               placeholder="Enter activity title">
                    </div>

                    <!-- Description -->
                    <div class="md:col-span-2">
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Description/Objective *</label>
                        <textarea name="description" required rows="4"
                                  class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-admin focus:border-transparent"
                                  placeholder="Enter activity description and objectives"></textarea>
                    </div>

                    <!-- Target Date -->
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Target Date *</label>
                        <input type="date" name="target_date" required 
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-admin focus:border-transparent">
                    </div>

                    <!-- Estimated Budget -->
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Estimated Budget (₱) *</label>
                        <input type="number" name="estimated_budget" required step="0.01" min="0"
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-admin focus:border-transparent"
                               placeholder="0.00">
                    </div>

                    <!-- Target Participants -->
                    <div>
                        <label class="block text-sm font-semibold text-gray-700 mb-2">Target Participants *</label>
                        <input type="number" name="target_participants" required min="1"
                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-admin focus:border-transparent"
                               placeholder="Number of participants">
                    </div>
                </div>

                <div class="flex justify-end gap-3">
                    <button type="reset" class="bg-gray-500 hover:bg-gray-600 text-white px-6 py-2 rounded-lg font-semibold transition">
                        <i class="fas fa-undo mr-2"></i>
                        Reset
                    </button>
                    <button type="submit" name="submit_gpb" class="bg-admin hover:bg-admin-dark text-white px-6 py-2 rounded-lg font-semibold transition btn-animate">
                        <i class="fas fa-save mr-2"></i>
                        Save GPB Entry
                    </button>
                </div>
            </form>
        </div>

        <!-- Year Filter -->
        <div class="bg-white rounded-xl shadow-lg p-6 mb-8">
            <form method="GET" class="flex gap-4 items-end">
                <div class="flex-1">
                    <label class="block text-sm font-semibold text-gray-700 mb-2">View GPB Entries for Year</label>
                    <select name="year" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-admin focus:border-transparent">
                        <?php for ($y = date('Y') + 5; $y >= 2020; $y--): ?>
                            <option value="<?php echo $y; ?>" <?php echo $y == $year_filter ? 'selected' : ''; ?>><?php echo $y; ?></option>
                        <?php endfor; ?>
                    </select>
                </div>
                <button type="submit" class="bg-admin hover:bg-admin-dark text-white px-6 py-2 rounded-lg font-semibold transition btn-animate">
                    <i class="fas fa-filter mr-2"></i>
                    Filter
                </button>
            </form>
        </div>

        <!-- Summary Cards -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div class="bg-white rounded-xl shadow-lg p-6">
                <div class="flex items-center">
                    <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center mr-4">
                        <i class="fas fa-list text-blue-600 text-xl"></i>
                    </div>
                    <div>
                        <div class="text-2xl font-bold text-gray-900"><?php echo count($gpb_entries); ?></div>
                        <div class="text-sm text-gray-600">Total Entries</div>
                    </div>
                </div>
            </div>
            <div class="bg-white rounded-xl shadow-lg p-6">
                <div class="flex items-center">
                    <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center mr-4">
                        <i class="fas fa-peso-sign text-green-600 text-xl"></i>
                    </div>
                    <div>
                        <div class="text-2xl font-bold text-gray-900">₱<?php echo number_format($total_budget, 0); ?></div>
                        <div class="text-sm text-gray-600">Total Budget</div>
                    </div>
                </div>
            </div>
            <div class="bg-white rounded-xl shadow-lg p-6">
                <div class="flex items-center">
                    <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center mr-4">
                        <i class="fas fa-users text-purple-600 text-xl"></i>
                    </div>
                    <div>
                        <div class="text-2xl font-bold text-gray-900"><?php echo number_format($total_participants); ?></div>
                        <div class="text-sm text-gray-600">Target Participants</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- GPB Entries List -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <div class="flex items-center justify-between mb-6">
                <h2 class="text-xl font-bold text-gray-900">
                    <i class="fas fa-table text-admin mr-2"></i>
                    GPB Entries for <?php echo $year_filter; ?>
                </h2>
                <a href="?year=<?php echo $year_filter; ?>&export=excel" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg font-semibold transition btn-animate">
                    <i class="fas fa-file-excel mr-2"></i>
                    Export to Excel
                </a>
            </div>

            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Activity Title</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Category</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Target Date</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Budget</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Participants</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($gpb_entries)): ?>
                            <tr>
                                <td colspan="6" class="px-6 py-8 text-center text-gray-500">
                                    <i class="fas fa-inbox text-4xl mb-3"></i>
                                    <p>No GPB entries found for <?php echo $year_filter; ?></p>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($gpb_entries as $entry): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4">
                                        <div class="text-sm font-semibold text-gray-900"><?php echo htmlspecialchars($entry['activity_title']); ?></div>
                                        <div class="text-xs text-gray-500"><?php echo htmlspecialchars(substr($entry['description'], 0, 60)); ?>...</div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-2 py-1 text-xs font-semibold rounded-full bg-purple-100 text-purple-800">
                                            <?php echo htmlspecialchars($entry['category']); ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700">
                                        <?php echo date('M d, Y', strtotime($entry['target_date'])); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-900">
                                        ₱<?php echo number_format($entry['estimated_budget'], 2); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700">
                                        <?php echo number_format($entry['target_participants']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-xs text-gray-500">
                                        <?php echo date('M d, Y', strtotime($entry['created_at'])); ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

    </div>
</div>

<?php include '../includes/footer.php'; ?>

