# PSAU GAD System - Troubleshooting Guide

## 🔧 Common Issues and Solutions

---

## 1. Permission Denied Errors

### Symptoms
```
Warning: mkdir(): Permission denied in /Applications/XAMPP/xamppfiles/htdocs/GAD/config/config.php
```

### Solution
```bash
# Navigate to project directory
cd /Applications/XAMPP/xamppfiles/htdocs/GAD

# Create directories manually
mkdir -p uploads/profiles uploads/certificates uploads/resources uploads/qr_codes logs

# Set proper permissions
chmod -R 777 uploads logs

# Or use the setup script
./setup.sh
```

---

## 2. Database Connection Failed

### Symptoms
- "Database connection failed" error
- Blank page when accessing the site
- PDO connection errors

### Solutions

**Check MySQL is running:**
1. Open XAMPP Control Panel
2. Verify MySQL has green "Running" status
3. If not, click "Start" button

**Verify database credentials:**
Edit `config/database.php`:
```php
define('DB_HOST', 'localhost');
define('DB_NAME', 'gad_system');
define('DB_USER', 'root');
define('DB_PASS', ''); // Usually empty for XAMPP
```

**Test MySQL connection:**
```bash
mysql -u root -e "SELECT 1"
```

**Create database if missing:**
```bash
mysql -u root < database/schema.sql
```

---

## 3. Class 'PHPMailer' Not Found

### Symptoms
```
Fatal error: Class 'PHPMailer\PHPMailer\PHPMailer' not found
```

### Solution
```bash
cd /Applications/XAMPP/xamppfiles/htdocs/GAD
composer install
```

**If Composer is not installed:**
```bash
# macOS
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# Verify
composer --version
```

---

## 4. Emails Not Sending

### Symptoms
- Registration emails not received
- No error messages
- Email log shows failures

### Solutions

**For Gmail:**

1. Enable 2-Factor Authentication
2. Generate App Password:
   - Go to: https://myaccount.google.com/security
   - Click "2-Step Verification"
   - Scroll to "App passwords"
   - Generate new password
   - Copy 16-character password

3. Update `config/config.php`:
```php
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_USERNAME', 'your-email@gmail.com');
define('SMTP_PASSWORD', 'your-16-char-app-password'); // NOT your regular password
define('SMTP_ENCRYPTION', 'tls');
```

**Test email manually:**
Create `test-email.php`:
```php
<?php
require_once 'config/config.php';
$email = new Email();
$result = $email->send('test@example.com', 'Test', 'This is a test email');
echo $result ? 'Email sent!' : 'Email failed!';
```

**Check logs:**
```bash
tail -f logs/error.log
```

---

## 5. Session Timeout / Not Logged In

### Symptoms
- Redirected to login page immediately
- "Session expired" messages
- Can't stay logged in

### Solutions

**Clear browser cookies:**
1. Press Ctrl+Shift+Delete (Cmd+Shift+Delete on Mac)
2. Clear cookies and cache
3. Try logging in again

**Check session configuration:**
Edit `config/config.php`:
```php
define('SESSION_TIMEOUT', 3600); // Increase to 7200 for 2 hours
```

**Verify session directory is writable:**
```bash
# Check PHP session path
php -i | grep session.save_path

# Make it writable
sudo chmod 777 /path/to/session/directory
```

**Restart Apache:**
1. Open XAMPP Control Panel
2. Stop Apache
3. Start Apache

---

## 6. Blank White Page

### Symptoms
- Page loads but shows nothing
- No error messages visible

### Solutions

**Enable error reporting:**
Edit `config/config.php` (temporarily):
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

**Check Apache error logs:**
```bash
# macOS XAMPP
tail -f /Applications/XAMPP/xamppfiles/logs/error_log

# Windows XAMPP
# Check: C:\xampp\apache\logs\error.log
```

**Check PHP errors:**
```bash
tail -f logs/error.log
```

**Common causes:**
- Missing semicolon in PHP
- Unclosed brackets
- Missing required files
- Database connection failed

---

## 7. CSS/Styling Not Loading

### Symptoms
- Page loads but no styling
- Plain HTML only
- Tailwind classes not working

### Solutions

**Check internet connection:**
- Tailwind CSS loads from CDN
- Requires active internet connection

**Clear browser cache:**
1. Press Ctrl+Shift+Delete
2. Clear cached images and files
3. Refresh page (Ctrl+F5)

**Check browser console:**
1. Press F12
2. Go to Console tab
3. Look for CDN loading errors

**Verify header.php is included:**
```php
<?php include 'includes/header.php'; ?>
```

---

## 8. 404 Not Found Errors

### Symptoms
- Pages show "404 Not Found"
- Links don't work
- Only index.php works

### Solutions

**Check file paths:**
- Verify file exists
- Check capitalization (case-sensitive on Linux)
- Use correct relative paths

**Enable mod_rewrite:**
Edit XAMPP `httpd.conf`:
```apache
LoadModule rewrite_module modules/mod_rewrite.so
```

**Verify .htaccess:**
Check `.htaccess` file exists in project root

**Restart Apache:**
After changing configuration, restart Apache in XAMPP

---

## 9. Upload Errors

### Symptoms
- "Failed to upload file"
- "Permission denied" on upload
- Files not saving

### Solutions

**Check directory permissions:**
```bash
chmod -R 777 uploads
ls -la uploads/
```

**Verify PHP upload settings:**
Edit `php.ini`:
```ini
upload_max_filesize = 10M
post_max_size = 10M
max_execution_time = 300
```

**Restart Apache after changing php.ini**

**Check upload directory exists:**
```bash
ls -la uploads/profiles
ls -la uploads/certificates
```

---

## 10. Database Import Errors

### Symptoms
- "Error importing SQL file"
- "Table already exists"
- "Syntax error in SQL"

### Solutions

**Drop existing database first:**
```sql
DROP DATABASE IF EXISTS gad_system;
```

**Import via command line:**
```bash
mysql -u root < database/schema.sql
```

**Import via phpMyAdmin:**
1. Go to http://localhost/phpmyadmin
2. Click "Import" tab
3. Choose `database/schema.sql`
4. Increase timeout if needed
5. Click "Go"

**Check for UTF-8 encoding:**
- Save SQL file as UTF-8
- Check for special characters

---

## 11. Composer Issues

### Symptoms
- "composer: command not found"
- "Failed to install dependencies"

### Solutions

**Install Composer:**
```bash
# macOS/Linux
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# Windows
# Download from: https://getcomposer.org/download/
```

**Update Composer:**
```bash
composer self-update
```

**Clear Composer cache:**
```bash
composer clear-cache
composer install
```

---

## 12. QR Code Not Generating

### Symptoms
- QR code images not showing
- Broken image icons
- QR code path errors

### Solutions

**Check internet connection:**
- QR codes use external API
- Requires internet access

**Verify QR codes directory:**
```bash
mkdir -p uploads/qr_codes
chmod 777 uploads/qr_codes
```

**Test QR generation:**
```php
<?php
require_once 'config/config.php';
$qr = new QRCode();
$code = $qr->generateForRegistration(1);
echo $code ? 'QR generated!' : 'QR failed!';
```

---

## 🔍 Debugging Tips

### Enable Debug Mode
Edit `config/config.php`:
```php
define('DEBUG_MODE', true);
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

### Check Error Logs
```bash
# Application logs
tail -f logs/error.log

# Apache logs
tail -f /Applications/XAMPP/xamppfiles/logs/error_log

# MySQL logs
tail -f /Applications/XAMPP/xamppfiles/logs/mysql_error.log
```

### Browser Developer Tools
1. Press F12
2. Check Console tab for JavaScript errors
3. Check Network tab for failed requests
4. Check Application tab for session/cookies

### Test Database Connection
Create `test-db.php`:
```php
<?php
require_once 'config/database.php';
try {
    $db = getDB();
    echo "Database connected successfully!";
} catch (Exception $e) {
    echo "Database connection failed: " . $e->getMessage();
}
```

---

## 📞 Getting Help

If issues persist:

1. **Check logs:**
   - `logs/error.log`
   - Apache error logs
   - Browser console

2. **Verify installation:**
   - All files present
   - Composer dependencies installed
   - Database created
   - Permissions correct

3. **Gather information:**
   - Error messages (exact text)
   - Screenshots
   - Steps to reproduce
   - PHP version: `php -v`
   - MySQL version: `mysql --version`

4. **Contact support:**
   - Email: gad@psau.edu.ph
   - Include error details and screenshots

---

## ✅ Health Check Script

Create `health-check.php`:
```php
<?php
echo "<h1>PSAU GAD System Health Check</h1>";

// PHP Version
echo "<p>PHP Version: " . phpversion() . "</p>";

// Database
try {
    require_once 'config/database.php';
    $db = getDB();
    echo "<p style='color:green'>✓ Database: Connected</p>";
} catch (Exception $e) {
    echo "<p style='color:red'>✗ Database: " . $e->getMessage() . "</p>";
}

// Directories
$dirs = ['uploads', 'uploads/profiles', 'uploads/certificates', 'logs'];
foreach ($dirs as $dir) {
    if (is_writable($dir)) {
        echo "<p style='color:green'>✓ Directory writable: $dir</p>";
    } else {
        echo "<p style='color:red'>✗ Directory not writable: $dir</p>";
    }
}

// Composer
if (file_exists('vendor/autoload.php')) {
    echo "<p style='color:green'>✓ Composer: Dependencies installed</p>";
} else {
    echo "<p style='color:red'>✗ Composer: Run 'composer install'</p>";
}
```

Access at: `http://localhost/GAD/health-check.php`

---

**Last Updated:** November 3, 2025

